const util = require("./util")

const { app, BrowserWindow, ipcMain, Menu, dialog } = require('electron')
const fs = require('fs')
const path = require('path')
const Datastore = require('nestdb')
const ip = require('ip')
// var Jimp = require('jimp');

const curiomaps = require('./curiomaps.json')
const langDefault = require('./lang_default.json')
let langCurrent = require('./lang_current.json')


const port = 3000

let cache = {
  teamIterator: 1,
  nodesAssigned: false
}

//#region Electron

// Keep a global reference of the window object, if you don't, the window will
// be closed automatically when the JavaScript object is garbage collected.
let mainWindow
let serverWindow


// DEPLOY === comment for Debug
dialog.showErrorBox = function (title, content) {
  if (debug) console.log(`${title}\n${content}`)
}

function createWindows() {

  const application = {
    label: 'Application',
    submenu: [
      {
        type: 'separator',
      },
      {
        label: 'Quit',
        accelerator: 'CmdOrCtrl+Q',
        click: () => {
          app.quit()
        },
      },
    ],
  }

  const edit = {
    label: 'Edit',
    submenu: [
      {
        label: 'Undo',
        accelerator: 'CmdOrCtrl+Z',
        role: 'undo',
      },
      {
        label: 'Redo',
        accelerator: 'Shift+CmdOrCtrl+Z',
        role: 'redo',
      },
      {
        type: 'separator',
      },
      {
        label: 'Cut',
        accelerator: 'CmdOrCtrl+X',
        role: 'cut',
      },
      {
        label: 'Copy',
        accelerator: 'CmdOrCtrl+C',
        role: 'copy',
      },
      {
        label: 'Paste',
        accelerator: 'CmdOrCtrl+V',
        role: 'paste',
      },
      {
        label: 'Select All',
        accelerator: 'CmdOrCtrl+A',
        role: 'selectAll',
      },
    ],
  }

  const template = [application, edit]

  // Comment out for debug
  Menu.setApplicationMenu(Menu.buildFromTemplate(template))
  


  // Create the browser window.
  mainWindow = new BrowserWindow({
    width: 640,
    height: 700,
    minWidth: 640,
    minHeight: 500,
    maxWidth: 800,
    icon: "./icon.ico",
    webPreferences: {
      nodeIntegration: true
    }
  })

  // DEPLOY === comment for Debug
  mainWindow.setMenuBarVisibility(false)


  serverWindow = new BrowserWindow({
    x: 0,
    y: 0,
    width: 400,
    height: 400,
    show: false, webPreferences: { nodeIntegration: true }
  })


  // and load the index.html of the app.
  mainWindow.loadFile('frontend/frontend.html')

  // Open the DevTools.
  // mainWindow.webContents.openDevTools()
  // serverWindow.webContents.openDevTools()


  // Emitted when the window is closed.
  mainWindow.on('closed', function () {
    // Dereference the window object, usually you would store windows
    // in an array if your app supports multi windows, this is the time
    // when you should delete the corresponding element.
    mainWindow = null
    serverWindow = null
    app.quit()
  })


  serverWindow.loadFile('server/server.html')




}



// This method will be called when Electron has finished
// initialization and is ready to create browser windows.
// Some APIs can only be used after this event occurs.
app.on('ready', createWindows)

// Quit when all windows are closed.
app.on('window-all-closed', function () {
  // On macOS it is common for applications and their menu bar
  // to stay active until the user quits explicitly with Cmd + Q
  // if (process.platform !== 'darwin') app.quit()
  app.quit()
})

app.on('activate', function () {
  // On macOS it's common to re-create a window in the app when the
  // dock icon is clicked and there are no other windows open.
  if (mainWindow === null) createWindows()
})

//#endregion

//#region Database Settings

db = {}

// DB-Class stores all possible classes
db.class = new Datastore({
  filename: path.resolve(__dirname, './dbclass.db'),
  autoload: true,
  onload: function (err) {
    if (err) {
      console.error('Failed to load the datastore:', err)
    } else {
      console.log('Loaded class db')
    }
  }
})

// DB-Scenario stores all possible Scenarios
db.scenario = new Datastore({
  filename: path.resolve(__dirname, './dbscenario.db'),
  autoload: true,
  onload: function (err) {
    if (err) {
      console.error('Failed to load the datastore:', err)
    } else {
      console.log('Loaded scenario db')
    }
  }
})



// db.scenario.ensureIndex({ fieldName: 'label', unique: true }, function (err) {
// })

db.game = {
  players: null,                          // will be DB for active players
  subtopics: null,                        // will be DB for current scenario
  questions: null,                        // will be DB for questions in current scenario
  scenarioMeta: {},                       // used for additional information for current the scenario
  state: 'Waiting',                       // game state for Unity
  timer: 0,                               // visible timer for Unity
  randSeed: util.randomInt(65500),        // seed for random functions in Unity
  nodeMap: curiomaps.default,             // loading node layout
  revealedNodes: [],                      // array of nodes that have been uncovered
  visitedNodes: [0],                      // array of nodes that have been visited
  endNode: null,                          // id of endnode
  playerPos: 0,                           // current node position of players
  endgameNodes: []                        // ids of subtopics that were featured in the endgame
  // teamPower: { '1': 1, '2': 1, '3': 1 }   // hitPower of each team
}


//#endregion


//#region Socket.IO Communication

ipcMain.on('ipcS2M_getGameDict', (event, arg) => {
  event.reply('ipcM2S_submitGameDict', langCurrent.game)
})

ipcMain.on('ipcS2M_loginStudent', (event, arg) => {

  db.game.players.findOne({ username: arg['username'] }, { username: 1, loggedin: 1, defaultteam: 1, team: 1 }, function (err, doc) {

    let outdoc = doc

    if (doc === null) {
      // Overwrite outdoc if user not found
      outdoc = { _id: "-1", username: "Username does not exist", team: 0 }
    } else if (doc.loggedin === false) {

      let teamAssignment = parseInt(doc.defaultteam)
      console.log(teamAssignment)

      if (!teamAssignment) {

        teamAssignment = cache.teamIterator
        cache.teamIterator += 1
        cache.teamIterator = cache.teamIterator > 3 ? 1 : cache.teamIterator
        console.log(teamAssignment)
      }

      db.game.players.update({ username: doc.username }, { $set: { loggedin: true, team: teamAssignment } }, {}, function (err, numReplaced) {
      })
      outdoc['team'] = teamAssignment
    }

    outdoc['gamestate'] = db.game.state
    console.log(outdoc)
    event.reply('ipcM2S_loginStudentReply', { data: outdoc, socketid: arg['socketid'] })

  })
})

//#endregion

//#region IPCs: General
ipcMain.on('ipcF2M_closeApp', (event, arg) => {
  console.log("CloseApp")
  app.quit()
})

ipcMain.on('ipcF2M_relaunchApp', (event, arg) => {
  app.relaunch()
  app.exit()
})

ipcMain.on('ipcSyncMF_getPublicUrl', (event, arg) => {
  event.returnValue = ip.address() + ":" + port
})

ipcMain.on('ipcF2M_setGameState', (event, arg) => {
  if (db.game.state != arg) {
    db.game.state = arg
    console.log("Game State set to " + arg)
    serverWindow.webContents.send('ipcM2S_gameStateChanged', arg)
  }
})

ipcMain.on('ipcF2M_setGameTimer', (event, arg) => {
  if (db.game.timer != arg) {
    db.game.timer = arg
    serverWindow.webContents.send('ipcM2S_timerChanged', arg)
  }
})

ipcMain.on('ipcS2M_getAssignedNodes', (event, arg) => {
  db.game.subtopics.find({}).sort({ node_id: 1 }).exec(function (err, docs) {
    serverWindow.webContents.send('ipcM2S_pushAssignedNodesToSockedID', {
      data: { randomSeed: db.game.randSeed, nodeData: docs },
      socketid: arg['socketid']
    })
  })
})

ipcMain.on('ipcF2M_requestScreenshot', (event, arg) => {

  db.game.players.find({ loggedin: true }, function (err, docs) {

    if (docs.length <= 0) return
    let randDoc = docs[util.randomInt(docs.length)]
    serverWindow.webContents.send('ipcM2S_requestScreenshot', randDoc.username)

  })

})


ipcMain.on('ipcS2M_submitScreenshot', (event, arg) => {
  mainWindow.webContents.send('ipcM2F_showScreenshot', arg)
})


//#endregion

//#region IPCs: Home

ipcMain.on('ipcSyncMF_listDbClasses', (event, arg) => {
  let output = []

  db.class.find({}, { label: 1, _id: 0 }, function (err, docs) {
    docs.forEach(element => {
      output.push(element['label'])
    })
    event.returnValue = output
  })
})

ipcMain.on('ipcSyncMF_listDbScenarios', (event, arg) => {
  let output = []
  db.scenario.find({}, { label: 1, _id: 0 }, function (err, docs) {
    docs.forEach(element => {
      output.push(element['label'])
    })
    event.returnValue = output
  })
})

ipcMain.on('ipcF2M_importClassFile', function (event, arg) {
  db.class.count({ label: arg['label'] }, function (err, count) {
    if (count === 0) {
      db.class.insert(arg, function (err, newDoc) {
        console.log('Imported class')
        event.reply('ipcM2F_refreshFrontend')
      })
    } else {
      event.reply('ipcM2F_dialogAlert', 'Class already exists.')
    }
  })
})

ipcMain.on('ipcF2M_importScenarioFile', function (event, arg) {
  db.scenario.count({ label: arg['label'] }, function (err, count) {
    if (count === 0) {
      db.scenario.insert(arg, function (err, newDoc) {
        console.log('Imported scenario')
        event.reply('ipcM2F_refreshFrontend')
      })
    } else {
      event.reply('ipcM2F_dialogAlert', 'Scenario already exists.')
    }
  })
})

ipcMain.on('ipcF2M_startGameSession_startServer', function (event, arg) {
  serverWindow.webContents.send('ipcM2S_startServer', port)

  // Init DB for players and scenario
  db.game.players = new Datastore()
  db.game.subtopics = new Datastore()
  db.game.questions = new Datastore()

  // Find a scenario in the database that matches the selection
  db.scenario.findOne({ label: arg['scenario'] }, function (err, doc) {
    db.game.scenarioMeta.label = doc['label']

    doc['subtopics'].forEach(item => {
      db.game.subtopics.insert({
        node_id: null,
        type: 'SUBTOPIC',
        subtopic: item['subtopic'],
        question: item['question'],
        correct: item['correct'],
        wrong: item['wrong']
      })
    })
  })

  db.class.findOne({ label: arg['class'] }, function (err, doc) {
    doc['students'].forEach(item => {
      db.game.players.insert({
        username: item['username'],
        defaultteam: item['defaultteam'],
        team: null,
        score: 0,
        loggedin: false,
        vote: null
      })
    })
  })

})

//#endregion

//#region IPCs: Manage Class
ipcMain.on('ipcSyncF2M_classExists', function (event, arg) {
  db.class.count({ label: arg }, function (err, count) {
    event.returnValue = count > 0
  })
})

ipcMain.on('ipcSyncF2M_getClass', function (event, arg) {
  db.class.findOne({ label: arg }, function (err, doc) {
    event.returnValue = doc
  })
})

ipcMain.on('ipcF2M_addEditClass', function (event, arg) {
  const mode = arg['mode']
  const data = arg['data']
  const id = arg['id']

  console.log(data)

  if (mode == 'edit') {
    db.class.update({ _id: id }, { $set: data }, {}, function (err, doc) {
      if (err) console.log(err)
    })
  } else if (mode == 'create') {
    db.class.insert(data, function (err, newDoc) {
      if (err) console.log(err)
    })
  } else if (mode == 'remove') {
    db.class.remove({ _id: id }, {}, function (err, numRemoved) {
      if (err) console.log(err)
    })
  }


})

ipcMain.on('ipcF2M_exportClassFile', function (event, arg) {
  // Find a scenario in the database that matches the selection
  db.class.findOne({ label: arg['class'] }, function (err, doc) {
    fs.writeFile(arg['path'], JSON.stringify(doc, null, 4), function (err) {
    })
  })
})

//#endregion

//#region IPCs: Manage Scenario
ipcMain.on('ipcSyncF2M_scenarioExists', function (event, arg) {
  db.scenario.count({ label: arg }, function (err, count) {
    event.returnValue = count > 0
  })
})

ipcMain.on('ipcSyncF2M_getScenario', function (event, arg) {
  db.scenario.findOne({ label: arg }, function (err, doc) {
    event.returnValue = doc
  })
})

ipcMain.on('ipcF2M_addEditScenario', function (event, arg) {
  const mode = arg['mode']
  const data = arg['data']
  const id = arg['id']

  if (mode == 'edit') {
    db.scenario.update({ _id: id }, { $set: data }, {}, function (err, doc) {
      if (err) console.log(err)
    })
  } else if (mode == 'create') {
    db.scenario.insert(data, function (err, newDoc) {
      if (err) console.log(err)
    })
  } else if (mode == 'remove') {
    db.scenario.remove({ _id: id }, {}, function (err, numRemoved) {
      if (err) console.log(err)
    })
  }


})



ipcMain.on('ipcF2M_exportScenarioFile', function (event, arg) {
  // Find a scenario in the database that matches the selection
  db.scenario.findOne({ label: arg['scenario'] }, function (err, doc) {
    fs.writeFile(arg['path'], JSON.stringify(doc, null, 4), function (err) {
    })
  })
})


//#endregion

//#region IPCs: Language
ipcMain.on('ipcSyncMF_getLanguage', (event, arg) => {
  event.returnValue = { current: langCurrent, default: langDefault }
})

ipcMain.on('ipcF2M_saveLanguage', function (event, arg) {
  fs.writeFileSync(path.resolve(__dirname, './lang_current.json'), JSON.stringify(arg, null, 4))
})

ipcMain.on('ipcF2M_exportLanguage', function (event, arg) {
  fs.writeFileSync(arg['path'], JSON.stringify(arg['data'], null, 4))
})



//#endregion

//#region IPCs: Game1 Waiting 

ipcMain.on('ipcF2M_getClassVars', (event, arg) => {
  db.game.players.find({}).sort({ loggedin: -1, team: 1, username: 1 }).exec(function (err, doc) {
    event.reply('ipcM2F_renderClassVars', doc)
  })
})

ipcMain.on('ipcF2M_assignTeamToStudent', (event, arg) => {
  console.log(`Place student into team: ${arg['id']} - ${arg['team']}`)
  db.game.players.update({ _id: arg['id'] }, { $set: { team: parseInt(arg['team']) } }, {}, function (err, numReplaced) {
  })
})


//#endregion

//#region IPCs: Game2 Beginning 


ipcMain.on('ipcF2M_pushPlayerInfoToClients', (event, arg) => {
  db.game.players.find({ loggedin: true }).sort({ team: 1, username: 1 }).exec(function (err, doc) {
    serverWindow.webContents.send('ipcM2S_pushPlayerInfo', doc)
  })
})



ipcMain.on('ipcF2M_assignNodes_pushToClients', (event, arg) => {

  if (!cache.nodesAssigned) {
    db.game.subtopics.find({}, function (err, docs) {
      // Create array from count, starting at 1
      let count = docs.length
      let numSeries = Array.from({ length: count }, (v, k) => k + 1)
      numSeries = util.shuffle(numSeries)

      docs.forEach(item => {
        db.game.subtopics.update({ _id: item['_id'] }, { $set: { node_id: numSeries.pop() } }, {})
      })

      db.game.endNode = count + 1
      db.game.subtopics.insert({ node_id: 0, type: 'START' })
      db.game.subtopics.insert({ node_id: db.game.endNode, type: 'END' })

      // Query again within the first query to get updated results
      // Push results to Clients
      db.game.subtopics.find({}).sort({ node_id: 1 }).exec(function (err, newdocs) {
        serverWindow.webContents.send('ipcM2S_pushAssignedNodesToClients', { randomSeed: db.game.randSeed, nodeData: newdocs })
      })
    })

    cache.nodesAssigned = true;
  } else {
    console.log("Nodes already assigned")

    db.game.subtopics.find({}).sort({ node_id: 1 }).exec(function (err, newdocs) {
      serverWindow.webContents.send('ipcM2S_pushAssignedNodesToClients', { randomSeed: db.game.randSeed, nodeData: newdocs })
    })
  }

})


//#endregion

//#region IPCs: Game3 Phase1 

ipcMain.on('ipcF2M_updateNodeLists_pushToClients', (event, arg) => {

  // ADD possible connections to revealed -- and remove duplicates
  db.game.revealedNodes = db.game.revealedNodes.concat(db.game.nodeMap[db.game.playerPos.toString()])
  db.game.revealedNodes = Array.from(new Set(db.game.revealedNodes))

  // limit selectables to id of end node
  db.game.revealedNodes = db.game.revealedNodes.filter(function (x) {
    return x <= db.game.endNode;
  })

  // Create selectable nodes by using revealed minus visited
  let selectableNodes = util.removeFromArray(db.game.revealedNodes, db.game.visitedNodes)

  serverWindow.webContents.send('ipcM2S_pushModifiedNodes', {
    revealed: db.game.revealedNodes,
    selectable: selectableNodes,
    visited: db.game.visitedNodes,
    forced: false
  })
})

ipcMain.on('ipcS2M_voteForNode', (event, arg) => {
  console.log(arg)
  db.game.players.update({ _id: arg['_id'] }, { $set: { vote: arg['vote'] } }, {}, function (err, numReplaced) { })
})

ipcMain.on('ipcF2M_resetVotes', (event, arg) => {
  db.game.players.update({ loggedin: true }, { $set: { vote: null } }, {}, function (err, numReplaced) {
    console.log(numReplaced)
  })
})

ipcMain.on('ipcF2M_notifyEndgameTriggered_pushToClients', (event, arg) => {

  // Overwrite selectable with endNode only
  serverWindow.webContents.send('ipcM2S_pushModifiedNodes', {
    revealed: db.game.revealedNodes,
    selectable: [db.game.endNode],
    visited: db.game.visitedNodes,
    forced: true
  })

})

//#endregion

//#region IPCs: Game4 Phase2

ipcMain.on('ipcF2M_forceEnd_pushToClients', (event, arg) => {
  let winningNode = db.game.endNode

  db.game.playerPos = winningNode
  db.game.visitedNodes.push(winningNode)
  serverWindow.webContents.send('ipcM2S_moveVoteResult', winningNode)
})


ipcMain.on('ipcF2M_tallyMoveVotes_pushToClients', (event, arg) => {

  // Tally Votes
  let candidates = util.removeFromArray(db.game.revealedNodes, db.game.visitedNodes)
  let candidateTally = {}

  console.log("CAN MOVE TO: ")
  console.log(candidates)

  db.game.players.find({}, function (err, docs) {
    docs.forEach(entry => {
      if (candidates.includes(entry['vote'])) {
        candidateTally[entry['vote']] = ~~candidateTally[entry['vote']] + 1
      }
    })

    let results = util.sortObjectParams(candidateTally).map(Number)
    results = results.reverse();

    // Check for winner -- if no votes were cast, pick one at random
    let winningNode = results.length > 0 ? results[0] : candidates[util.randomInt(candidates.length)]

    // Set new playerPos
    db.game.playerPos = winningNode

    // Update visited Nodes by the winning Node
    db.game.visitedNodes.push(winningNode)

    // Inform about result
    mainWindow.webContents.send('ipcM2F_moveVoteResults', { tally: candidateTally, winner: winningNode, endnode: db.game.endNode })
    serverWindow.webContents.send('ipcM2S_moveVoteResult', winningNode)

  })
})

//#endregion

//#region IPCs: Game5 Phase34

ipcMain.on('ipcS2M_submitQuestion', (event, arg) => {

  db.game.questions.insert({
    userid: arg.userid,
    username: arg.username,
    team: arg.team,
    atnode: db.game.playerPos,
    state: 'UNCHECKED',
    question: arg.question
  })

})

ipcMain.on('ipcF2M_getUncheckedQuestions', (event, arg) => {
  db.game.questions.find({ state: 'UNCHECKED' }, { username: 1, question: 1 }, function (err, docs) {
    if (docs.length > 0) {
      event.reply('ipcM2F_checkAttachedQuestions', docs)
    }
  })
})

ipcMain.on('ipcF2M_updateQuestionState', (event, arg) => {

  // Update question state
  db.game.questions.update({ _id: arg.id }, { $set: { state: arg.state } }, {}, function (err, numReplaced) {
    console.log("Updated " + numReplaced + " -- is " + arg.state)
  })

  if (arg.state === 'ACCEPTED') {
    db.game.players.update({ username: arg.username }, { $inc: { score: 1 } }, {}, function (err, numReplaced) { })
  }

})

ipcMain.on('ipcF2M_getAcceptedQuestions_pushToClients', (event, arg) => {

  db.game.questions.find({ atnode: db.game.playerPos, state: 'ACCEPTED' }, { _id: 0, username: 1, question: 1 }, function (err, docs) {
    if (docs.length > 0) {
      console.log(docs)
      serverWindow.webContents.send('ipcM2S_listAcceptedQuestions', docs)
    }
  })
})

//#endregion

//#region IPCs: Game6 Phase5

ipcMain.on('ipcF2M_evaluateTeams_pushToClients', (event, arg) => {

  function EvaluateTeams(doc) {
    let teamList = [1, 2, 3]
    let results = {}

    let count = 0
    teamList.forEach(teamEntry => {

      db.game.questions.find({
        atnode: db.game.playerPos,
        team: teamEntry,
        state: { $in: ['ACCEPTED'] }
      }, {}, function (err, docs) {

        // Results for Student
        let evalResult = {}
        evalResult.passed = false
        evalResult.question = ""
        evalResult.answer = ""

        console.log(`Team ${teamEntry} provided ${docs.length} accepted questions`)

        if (docs.length > 2) {
          // PASSED THE THRESHOLD
          evalResult.passed = true
          evalResult.question = doc.question
          evalResult.answer = doc.correct
        }

        results[teamEntry] = evalResult
        count++

        if (count >= 3) {
          // Send results once all teams have been processed
          console.log(results)
          serverWindow.webContents.send('ipcM2S_teamEvaluation', results)
        }

      })
    })
  }


  db.game.subtopics.findOne({ node_id: db.game.playerPos }, function (err, doc) {
    console.log(doc)
    EvaluateTeams(doc)
  })

})

//#endregion

//#region IPCs: Endgame2

ipcMain.on('ipcF2M_endgamePickSubtopic_pushToClients', (event, arg) => {
  // Ignore start, end, and any picked endgame nodes
  let possibleNodes = db.game.visitedNodes.filter(node => {
    return (node !== 0) && (node !== db.game.endNode) && !db.game.endgameNodes.includes(node)
  })

  if (possibleNodes.length < 1) {
    possibleNodes = [...Array(db.game.endNode - 1)].map((_, i) => i + 1)
  }

  // Pick random node from possible nodes and append to picked array
  let pickId = util.randomInt(possibleNodes.length)
  let pickedNode = possibleNodes[pickId]

  db.game.endgameNodes.push(pickedNode)

  db.game.subtopics.findOne({ node_id: pickedNode }, function (err, doc) {
    console.log(doc)
    serverWindow.webContents.send('ipcM2S_endgameSubtopic', doc)


  })

})



//#endregion


//#region IPCs: Endgame3

ipcMain.on('ipcF2M_endgameRoundResults_pushToClients', async (event, arg) => {

  // Add score to players for correct answer
  function RewardCorrectAnswers() {
    return new Promise(resolve => {

      // Where vote equals 1, increase score by 5
      db.game.players.update({ vote: 1 }, { $inc: { score: 5 } }, {}, function (err, numReplaced) {
        resolve()
      })
    })
  }

  function CountTeamSize(teamId) {
    return new Promise(resolve => {
      console.log(`TeamId is ${teamId}`)
      db.game.players.find({ team: teamId, loggedin: true }, function (err, docs) {
        console.log(docs)
        console.log(`Team size is ${docs.length}`)
        resolve(docs.length)
      })
    })
  }

  function CountCorrectVotes(teamId) {
    return new Promise(resolve => {

      db.game.players.find({ team: teamId, loggedin: true, vote: 1 }, function (err, docs) {

        console.log(`Correct votes ${docs.length}`)
        resolve(docs.length)
      })
    })
  }

  function GetCurrentSubtopic() {
    return new Promise(resolve => {
      const currNode = db.game.endgameNodes[db.game.endgameNodes.length - 1]

      db.game.subtopics.findOne({ node_id: currNode }, function (err, doc) {
        resolve(doc)
      })
    })
  }

  await RewardCorrectAnswers()

  let teamList = [1, 2, 3]
  let results = {}

  for (const teamEntry of teamList) {
    const teamSize = await CountTeamSize(teamEntry)
    const teamVotes = await CountCorrectVotes(teamEntry)
    const teamScore = teamSize > 0 ? teamVotes / teamSize : 0
    results[teamEntry] = teamScore
  }


  const subtopic = await GetCurrentSubtopic()

  console.log(results)
  console.log(subtopic)


  serverWindow.webContents.send('ipcM2S_endgameRoundResults', {
    results: results,
    subtopic: subtopic
  })


})



//#endregion



//#region IPCs: Endgame5

ipcMain.on('ipcF2M_endgameInitStickerBoard_pushToClients', (event, arg) => {
  db.game.players.find({ loggedin: true }, { _id: 0, username: 1, score: 1 }, function (err, docs) {
    console.log(docs)
    serverWindow.webContents.send('ipcM2S_playerScores', docs)
  })

})


ipcMain.on('ipcS2M_submitSticker', (event, arg) => {
  serverWindow.webContents.send('ipcM2S_stickerPlacement', arg)
})

ipcMain.on('ipcF2M_stopStickerPlacement', (event, arg) => {
  serverWindow.webContents.send('ipcM2S_stopStickerPlacement')
})


// ipcMain.on('ipcF2M_endGameSession', (event, arg) => {
//   serverWindow.webContents.send('ipcM2S_endGameSession')
// })

//#endregion
